/*
 * Demo ET-ESP32-RS485 Hardware Board
 * MCU      : ESP32-WROVER
 *          : Arduino Tools Board : ESP32 Wrover Module
 * Software : https://dl.espressif.com/dl/package_esp32_index.json        
 * Reserve  : Boot Config       
 *          : -> IO0(BOOT)
 *          : -> IO2(Don't Card on 3V3V Version)
 *          : -> IO5
 *          : -> IO12(MTDI)
 *          : -> IO15(MTDO)
 * Reserve  : SFLASH        
 *          : -> CMD
 *          : -> CLK
 *          : -> SD0
 *          : -> SD1
 *          : -> SD2
 *          : -> SD3
 * Debug    : Serial0 : USART0(USB)        
 *          : -> RX0(Debug & Download:IO3)
 *          : -> TX0(Debug & Download:IO1)
 * NB-IoT   : Serial1 : SIM7020E(BK-7020 V2)       
 *          : -> RX1(T:IO14)
 *          : -> TX1(R:IO13)
 *          : -> PWRKEY(K:IO33)
 *          : -> SLEEP(S:IO32)
 * RS485    : Serial1 : RS485  
 *          : -> RX2(IO26)
 *          : -> TX2(IO27)
 *          : -> DIR(Direction : IO25 : LOW = Receive, HIGH = Send)
 * I2C#1    : I2C BUS
 *          : -> SCL1(IO22)
 *          : -> SDA1(IO21)
 * RTC      : -> RTC:DS3231/DS3232
 *          : -> PCF8574/A(Relay8)
 *          : -> PCF8574/A(DC-IN8)
 * LED      : LED Status(Active High)
 *          : -> LED0(IO2)
 * Demo     : Interface ET-I2C LONGLENGTH REPEATER(PCF8574A)         
 */
 
 
//=================================================================================================
#include <Wire.h> 
#include "ETT_PCF8574.h"
#include "ETT_PCA9548.h"
//=================================================================================================
//=================================================================================================
#include <Adafruit_SHT31.h>
//=================================================================================================

//=================================================================================================
// Start of Default Hardware : ET-ESP32-RS485
//=================================================================================================
// Remap Pin USART -> C:\Users\Admin\Documents\Arduino\hardware\espressif\esp32\cores\esp32\HardwareSerial.cpp
//                    C:\Users\Admin\AppData\Local\Arduino15\packages\esp32\hardware\esp32\1.0.0\cores\esp32\HardwareSerial.cpp
//=================================================================================================
#include <HardwareSerial.h>
//=================================================================================================
#define SerialDebug  Serial                                                                       // USB Serial(Serial0)
//=================================================================================================
#define SerialNBIOT_RX_PIN    14
#define SerialNBIOT_TX_PIN    13
#define SerialNBIOT  Serial1                                                                      // Serial1(IO13=TXD,IO14=RXD)
//=================================================================================================
#define SerialRS485_RX_PIN    26
#define SerialRS485_TX_PIN    27
#define SerialRS485  Serial2                                                                      // Serial2(IO27=TXD,IO26=RXD)
//=================================================================================================
#define RS485_DIRECTION_PIN   25                                                                  // ESP32-WROVER :IO25
#define RS485_RXD_SELECT      LOW
#define RS485_TXD_SELECT      HIGH
//=================================================================================================
#define SIM7020E_PWRKEY_PIN   33                                                                  // ESP32-WROVER :IO33
#define SIM7020E_SLEEP_PIN    32                                                                  // ESP32-WROVER :IO32
#define SIM7020E_PWRKEY_LOW   LOW                                                                 // Start Power-ON
#define SIM7020E_PWRKEY_HIGH  HIGH                                                                // Release Signal
#define SIM7020E_SLEEP_LOW    LOW                                                                 // Pull-Up DTR to Enable Sleep
#define SIM7020E_SLEEP_HIGH   HIGH                                                                // DTR=Low(Wakeup From Sleep)
//=================================================================================================
#define I2C_SCL1_PIN          22                                                                  // ESP32-WROVER : IO22(SCL1)
#define I2C_SDA1_PIN          21                                                                  // ESP32-WROVER : IO21(SDA1)
//=================================================================================================

//=================================================================================================
#define LED_PIN               2                                                                   // ESP-WROVER  : IO2
#define LedON                 1
#define LedOFF                0
//=================================================================================================
#define CS_SD_CARD_PIN        4                                                                   // ESP-WROVER  : IO4
#define SD_CARD_DISABLE       1
#define SD_CARD_ENABLE        0
//=================================================================================================
// End of Default Hardware : ET-ESP32-RS485
//=================================================================================================

//=================================================================================================
ETT_PCF8574 master_relay(PCF8574_ID_DEV0);        // Master : Internal Relay
ETT_PCF8574 master_input(PCF8574A_ID_DEV0);       // Master : Internal OPTO-Input
//=================================================================================================

//=================================================================================================
ETT_PCA9548 i2c_select0(PCA9548_ID_DEV0);         // Slave Select Device[0]
ETT_PCF8574 slave_unit0(PCF8574A_ID_DEV7);        // Slave Unit[0] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select1(PCA9548_ID_DEV1);         // Slave Select Device[1]
ETT_PCF8574 slave_unit1(PCF8574A_ID_DEV7);        // Slave Unit[1] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select2(PCA9548_ID_DEV2);         // Slave Select Device[2]
ETT_PCF8574 slave_unit2(PCF8574A_ID_DEV7);        // Slave Unit[2] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select3(PCA9548_ID_DEV3);         // Slave Select Device[3]
ETT_PCF8574 slave_unit3(PCF8574A_ID_DEV7);        // Slave Unit[3] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select4(PCA9548_ID_DEV4);         // Slave Select Device[4]
ETT_PCF8574 slave_unit4(PCF8574A_ID_DEV7);        // Slave Unit[4] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select5(PCA9548_ID_DEV5);         // Slave Select Device[5]
ETT_PCF8574 slave_unit5(PCF8574A_ID_DEV7);        // Slave Unit[5] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select6(PCA9548_ID_DEV6);         // Slave Select Devic[6]
ETT_PCF8574 slave_unit6(PCF8574A_ID_DEV7);        // Slave Unit[6] (Relay & OPTO-Input)
//=================================================================================================
ETT_PCA9548 i2c_select7(PCA9548_ID_DEV7);         // Slave Select Device[7]
ETT_PCF8574 slave_unit7(PCF8574A_ID_DEV7);        // Slave Unit[7] (Relay & OPTO-Input)
//=================================================================================================
uint8_t state;
unsigned long lastGetDelayTime = 0;
//=================================================================================================

//=================================================================================================
Adafruit_SHT31 sht31_dev0 = Adafruit_SHT31();                                                    // I2C
Adafruit_SHT31 sht31_dev1 = Adafruit_SHT31();                                                    // I2C
//=================================================================================================
float sht31_dev0_temperature;
float sht31_dev0_humidity;
bool  sht31_dev0_status;
float sht31_dev1_temperature;
float sht31_dev1_humidity;
bool  sht31_dev1_status;
//=================================================================================================
unsigned long lastGetI2CSensorTime = 0;
//=================================================================================================
  
void setup() 
{
  //===============================================================================================
  // Start of Initial Default Hardware : ET-ESP32-RS485
  //===============================================================================================
  pinMode(LED_PIN, OUTPUT);
  digitalWrite(LED_PIN, LedOFF);
  //===============================================================================================
  pinMode(CS_SD_CARD_PIN, OUTPUT);
  digitalWrite(CS_SD_CARD_PIN, SD_CARD_DISABLE);
  //===============================================================================================
  Wire.begin(I2C_SDA1_PIN,I2C_SCL1_PIN);                                                      
  //===============================================================================================
  pinMode(SIM7020E_PWRKEY_PIN,OUTPUT);                                                           // PWRKEY
  pinMode(SIM7020E_SLEEP_PIN,OUTPUT);                                                            // SLEEP
  //===============================================================================================
  digitalWrite(SIM7020E_SLEEP_PIN,SIM7020E_SLEEP_LOW);                                           // Wakeup From Sleep By Pullup DTR = HIGH
  //=============================================================================================== 
  digitalWrite(SIM7020E_PWRKEY_PIN,SIM7020E_PWRKEY_LOW);                                         // PWRKEY = LOW(Active Power-ON:800mS++)
  delay(1000);
  digitalWrite(SIM7020E_PWRKEY_PIN,SIM7020E_PWRKEY_HIGH);                                        // PWRKEY = HIGH(Ready) 
  //===============================================================================================
  pinMode(RS485_DIRECTION_PIN, OUTPUT);                                                          // RS485 Direction
  digitalWrite(RS485_DIRECTION_PIN, RS485_RXD_SELECT);
  //===============================================================================================
  SerialDebug.begin(115200);
  while(!SerialDebug);
  //===============================================================================================
  SerialNBIOT.begin(115200);
  while(!SerialNBIOT);
  //===============================================================================================
  SerialRS485.begin(115200);
  while(!SerialRS485);
  //===============================================================================================
  // End of Initial Default Hardware : ESP32
  //===============================================================================================

  //===============================================================================================
  SerialDebug.println();
  SerialDebug.println("ET-ESP32-RS485...Default Hardware Config...Complete");
  SerialDebug.println("USART0 = SerialDebug(USB)");
  SerialDebug.println("USART1 = SerialSIM7020E(NB-IoT)");
  SerialDebug.println("USART2 = SerialRS485(RJ45)");
  //===============================================================================================
  
  //===============================================================================================
  i2c_select0.begin();            // Initial ET-I2C Longlngth Repeater : Slave[0]
  i2c_select0.disableBus();
  //===============================================================================================
  i2c_select1.begin();            // Initial ET-I2C Longlngth Repeater : Slave[1]
  i2c_select1.disableBus();
  //===============================================================================================
  i2c_select2.begin();            // Initial ET-I2C Longlngth Repeater : Slave[2]
  i2c_select2.disableBus();
  //===============================================================================================
  i2c_select3.begin();            // Initial ET-I2C Longlngth Repeater : Slave[3]
  i2c_select3.disableBus();
  //===============================================================================================
  i2c_select4.begin();            // Initial ET-I2C Longlngth Repeater : Slave[4]
  i2c_select4.disableBus();
  //===============================================================================================
  i2c_select5.begin();            // Initial ET-I2C Longlngth Repeater : Slave[5]
  i2c_select5.disableBus();
  //===============================================================================================
  i2c_select6.begin();            // Initial ET-I2C Longlngth Repeater : Slave[6]
  i2c_select6.disableBus();
  //===============================================================================================
  i2c_select7.begin();            // Initial ET-I2C Longlngth Repeater : Slave[7]
  i2c_select7.disableBus();
  //===============================================================================================


  //===============================================================================================
  master_relay.begin(0xFF);
  master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
  master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
  //===============================================================================================
  master_input.begin(0xFF);
  //===============================================================================================
  i2c_select0.enableBus();                                          // Enable BUS
  slave_unit0.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  //===============================================================================================
  SerialDebug.println("SHT31 I2C(0x44) Test");
  //===============================================================================================
  sht31_dev0_status = sht31_dev0.begin(0x44);
  //===============================================================================================
  if(!sht31_dev0_status)
  {
      SerialDebug.println("Initial SHT31(0)...Error");
  }
  //===============================================================================================
  SerialDebug.println("Initial SHT31(0)...Complete");
  //===============================================================================================
  i2c_select0.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select1.enableBus();                                          // Enable BUS
  slave_unit1.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  //===============================================================================================
  SerialDebug.println("SHT31 I2C(0x44) Test");
  //===============================================================================================
  sht31_dev1_status = sht31_dev1.begin(0x44);
  //===============================================================================================
  if(!sht31_dev1_status)
  {
      SerialDebug.println("Initial SHT31(1)...Error");
  }
  //===============================================================================================
  SerialDebug.println("Initial SHT31(1)...Complete");
  //===============================================================================================
  i2c_select1.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select2.enableBus();                                          // Enable BUS
  slave_unit2.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  i2c_select2.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select3.enableBus();                                          // Enable BUS
  slave_unit3.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  i2c_select3.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select4.enableBus();                                          // Enable BUS
  slave_unit4.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  i2c_select4.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select5.enableBus();                                          // Enable BUS
  slave_unit5.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  i2c_select5.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select6.enableBus();                                          // Enable BUS
  slave_unit6.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  i2c_select6.disableBus();                                         // Disable BUS
  //===============================================================================================
  i2c_select7.enableBus();                                          // Enable BUS
  slave_unit7.begin(0xFF);                                          // Initial PCF8574 Control Relay[0,1]
  slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);            // Control Relay[0]
  slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);            // Control Relay[1]
  i2c_select7.disableBus();                                         // Disable BUS
  //===============================================================================================
  
  //=============================================================================================== 
  state = 12;
  //=============================================================================================== 

  //=============================================================================================== 
  SerialDebug.println();
  //=============================================================================================== 
  if(master_relay.lastError()==PCF8574_OK)
  {
    Serial.println("master output...ready");
  }
  else
  {
    Serial.println("master output...error");
  }
  //=============================================================================================== 
  if(master_input.lastError()==PCF8574_OK)
  {
    Serial.println("master input....ready");
  }
  else
  {
    Serial.println("master input....error");
  }
  //=============================================================================================== 
  
  //=============================================================================================== 
  if(i2c_select0.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select0.....ready");
    //============================================================================================= 
    i2c_select0.enableBus();
    //=============================================================================================
    slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select0.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select0.....error");
  }
  //=============================================================================================== 

  //=============================================================================================== 
  if(i2c_select1.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select1.....ready");
    //============================================================================================= 
    i2c_select1.enableBus();
    //=============================================================================================
    slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select1.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select1.....error");
  }
  //=============================================================================================== 

  //=============================================================================================== 
  if(i2c_select2.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select2.....ready");
    //============================================================================================= 
    i2c_select2.enableBus();
    //=============================================================================================
    slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select2.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select2.....error");
  }
  //=============================================================================================== 
  
  //=============================================================================================== 
  if(i2c_select3.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select3.....ready");
    //============================================================================================= 
    i2c_select3.enableBus();
    //=============================================================================================
    slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select3.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select3.....error");
  }
  //=============================================================================================== 

  //=============================================================================================== 
  if(i2c_select4.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select4.....ready");
    //============================================================================================= 
    i2c_select4.enableBus();
    //=============================================================================================
    slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select4.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select4.....error");
  }
  //=============================================================================================== 

  //=============================================================================================== 
  if(i2c_select5.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select5.....ready");
    //============================================================================================= 
    i2c_select5.enableBus();
    //=============================================================================================
    slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select5.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select5.....error");
  }
  //=============================================================================================== 


  //=============================================================================================== 
  if(i2c_select6.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select6.....ready");
    //============================================================================================= 
    i2c_select6.enableBus();
    //=============================================================================================
    slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select6.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select6.....error");
  }
  //=============================================================================================== 

  //=============================================================================================== 
  if(i2c_select7.lastError()==PCA9548_OK)
  {
    Serial.println("i2c select7.....ready");
    //============================================================================================= 
    i2c_select7.enableBus();
    //=============================================================================================
    slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
    slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
    //============================================================================================= 
    i2c_select7.disableBus();
    //============================================================================================= 
  }
  else
  {
    Serial.println("i2c select7.....error");
  }
  //===============================================================================================  
}

void loop() 
{
  //===============================================================================================
  // Start of Read & Write I2C In/Out PCF8574
  //===============================================================================================
  if(millis() - lastGetDelayTime > 50ul)                                                       // 0.5-Second
  {
    //=============================================================================================
    state++;
    if(state > 16) state = 0;                                                                    // 0,1,...,15
    //=============================================================================================
    switch(state) 
    {
      case 0: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;
      
      case 1: 
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        i2c_select7.disableBus();
        //=========================================================================================
      break;
      
      case 2: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 3: 
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        i2c_select6.disableBus();
        //========================================================================================= 
      break;
      
      case 4: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 5: 
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        i2c_select5.disableBus();
        //========================================================================================= 
      break;
      
      case 6: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 7: 
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        i2c_select4.disableBus();
        //========================================================================================= 
      break;
      
      case 8: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 9: 
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        i2c_select3.disableBus();
        //========================================================================================= 
      break;
      
      case 10: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 11: 
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        i2c_select2.disableBus();
        //========================================================================================= 
      break;
      
      case 12: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_ON);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_OFF);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 13:
        //========================================================================================= 
        i2c_select1.enableBus();
        //========================================================================================= 
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        //=========================================================================================
        sht31_dev1_temperature  = sht31_dev1.readTemperature();                                             // *C
        sht31_dev1_humidity     = sht31_dev1.readHumidity();                                                // %RH
        //=========================================================================================
        if(! isnan(sht31_dev1_temperature)) 
        {
          SerialDebug.print("SHT31(1) Temperature = ");
          SerialDebug.print(sht31_dev1_temperature);
          SerialDebug.print("*C");
        }
        else
        {
          SerialDebug.println("SHT31(1) Temperature = Error  ");
        }
        if(! isnan(sht31_dev1_humidity)) 
        {
          SerialDebug.print(" : Humidity = ");
          SerialDebug.print(sht31_dev1_humidity);
          SerialDebug.println("%RH");
        }
        else
        {
          SerialDebug.println("SHT31(1) Humidity = Error");
        }
        //========================================================================================= 
        i2c_select1.disableBus();
        //=========================================================================================  
      break;
      
      case 14: 
        //=========================================================================================
        master_relay.writePin(MASTER_RELAY_OUT7_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT6_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT5_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT4_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT3_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT2_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT1_PIN, RELAY_OFF);
        master_relay.writePin(MASTER_RELAY_OUT0_PIN, RELAY_ON);
        //=========================================================================================
        i2c_select7.enableBus();
        slave_unit7.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit7.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select7.disableBus();
        //========================================================================================= 
        i2c_select6.enableBus();
        slave_unit6.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit6.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select6.disableBus();
        //========================================================================================= 
        i2c_select5.enableBus();
        slave_unit5.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit5.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select5.disableBus();
        //========================================================================================= 
        i2c_select4.enableBus();
        slave_unit4.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit4.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select4.disableBus();
        //========================================================================================= 
        i2c_select3.enableBus();
        slave_unit3.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit3.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select3.disableBus();
        //========================================================================================= 
        i2c_select2.enableBus();
        slave_unit2.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit2.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select2.disableBus();
        //========================================================================================= 
        i2c_select1.enableBus();
        slave_unit1.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit1.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select1.disableBus();
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_ON);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_OFF);
        i2c_select0.disableBus();
        //========================================================================================= 
      break;

      case 15: 
        //========================================================================================= 
        i2c_select0.enableBus();
        slave_unit0.writePin(SLAVE_RELAY_OUT1_PIN, RELAY_OFF);
        slave_unit0.writePin(SLAVE_RELAY_OUT0_PIN, RELAY_ON);
        //=========================================================================================
        sht31_dev0_temperature  = sht31_dev0.readTemperature();                                  // *C
        sht31_dev0_humidity     = sht31_dev0.readHumidity();                                     // %RH
        //=========================================================================================
        if(! isnan(sht31_dev0_temperature)) 
        {
          SerialDebug.print("SHT31(0) Temperature = ");
          SerialDebug.print(sht31_dev0_temperature);
          SerialDebug.print("*C");
        }
        else
        {
          SerialDebug.println("SHT31(0) Temperature = Error  ");
        }
        if(! isnan(sht31_dev0_humidity)) 
        {
          SerialDebug.print(" : Humidity = ");
          SerialDebug.print(sht31_dev0_humidity);
          SerialDebug.println("%RH");
        }
        else
        {
          SerialDebug.println("SHT31(0) Humidity = Error");
        }
        //========================================================================================= 
        i2c_select0.disableBus();
        //========================================================================================= 
      break;
    }
    //=============================================================================================
    delay(10);                                                                                   //Wait Output Relay On Input Opto Complete
    //=============================================================================================
    Serial.print("Master IN = ");
    //=============================================================================================
    Serial.print(master_input.readPin(MASTER_OPTO_IN7_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN6_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN5_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN4_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN3_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN2_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN1_PIN));
    Serial.print(master_input.readPin(MASTER_OPTO_IN0_PIN));
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select7.enableBus();
    Serial.print(slave_unit7.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit7.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select7.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select6.enableBus();
    Serial.print(slave_unit6.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit6.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select6.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select5.enableBus();
    Serial.print(slave_unit5.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit5.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select5.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select4.enableBus();
    Serial.print(slave_unit4.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit4.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select4.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select3.enableBus();
    Serial.print(slave_unit3.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit3.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select3.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select2.enableBus();
    Serial.print(slave_unit2.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit2.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select2.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select1.enableBus();
    Serial.print(slave_unit1.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit1.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select1.disableBus();
    //=============================================================================================
    Serial.print(" -> ");
    //=============================================================================================
    i2c_select0.enableBus();
    Serial.print(slave_unit0.readPin(SLAVE_OPTO_IN1_PIN));
    Serial.print(slave_unit0.readPin(SLAVE_OPTO_IN0_PIN));
    i2c_select0.disableBus();
    //=============================================================================================
    Serial.println();
    //=============================================================================================
    lastGetDelayTime = millis();
    //=============================================================================================  
  }
  //===============================================================================================
  // End of Read & Write I2C In/Out PCF8574
  //===============================================================================================
  
}






